<?php
/**
 * Add extra profile fields for users in admin
 *
 * @package cardealer-helper/classes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if ( ! class_exists( 'CarDealer_Admin_Profile', false ) ) :

	/**
	 * CarDealer_Admin_Profile Class.
	 */
	class CarDealer_Admin_Profile {

		/**
		 * Fields
		 */
		public $fields = array();

		/**
		 * Hook in tabs.
		 */
		public function __construct() {
			add_action( 'show_user_profile', array( $this, 'add_meta_fields' ) );
			add_action( 'edit_user_profile', array( $this, 'add_meta_fields' ) );

			add_action( 'personal_options_update', array( $this, 'save_meta_fields' ) );
			add_action( 'edit_user_profile_update', array( $this, 'save_meta_fields' ) );

			add_filter( 'cardealer_user_profile_meta_fields', array( $this, 'social_profile_fields' ) );
			add_filter( 'cardealer_user_profile_meta_fields_social_profiles', array( $this, 'social_profiles' ), 5 );

			add_action( 'wp_ajax_profile_image_upload', array( $this, 'profile_image_upload_callback' ) );
			add_action( 'wp_ajax_nopriv_profile_image_upload', array( $this, 'profile_image_upload_callback' ) );
		}

		/**
		 * Show Address Fields on edit user pages.
		 *
		 * @param WP_User $user
		 */
		public function add_meta_fields( $user ) {
			// Check that the current user have the capability to edit the $user_id
			if ( ! apply_filters( 'cardealer_current_user_can_edit_user_profile_meta_fields', current_user_can( 'edit_user' ), $user->ID ) ) {
				return;
			}

			$fieldsets = $this->get_fields();

			foreach ( $fieldsets as $fieldset_key => $fieldset ) {

				// Skip, if fields are empty.
				if ( empty( $fieldset['fields'] ) ) {
					continue;
				}

				$fields = $fieldset['fields'];
				?>
				<h2><?php echo $fieldset['title']; ?></h2>
				<table id="<?php echo esc_attr( 'fieldset-' . $fieldset_key ); ?>" class="form-table cardealer-profile-fields">
					<?php
					foreach ( $fields as $key => $field ) {
						?>
						<tr>
							<th>
								<label for="<?php echo esc_attr( $key ); ?>"><?php echo esc_html( $field['label'] ); ?></label>
							</th>
							<td>
								<?php
								if ( ! empty( $field['type'] ) && 'select' === $field['type'] ) {
									?>
									<select name="<?php echo esc_attr( $key ); ?>" id="<?php echo esc_attr( $key ); ?>" class="<?php echo esc_attr( $field['class'] ); ?>" style="width: 25em;">
										<?php
										$selected = esc_attr( get_user_meta( $user->ID, $key, true ) );
										foreach ( $field['options'] as $option_key => $option_value ) :
											?>
											<option value="<?php echo esc_attr( $option_key ); ?>" <?php selected( $selected, $option_key, true ); ?>><?php echo esc_html( $option_value ); ?></option>
											<?php
										endforeach;
										?>
									</select>
									<?php
								} elseif ( ! empty( $field['type'] ) && 'checkbox' === $field['type'] ) {
									?>
									<input type="checkbox" name="<?php echo esc_attr( $key ); ?>" id="<?php echo esc_attr( $key ); ?>" value="1" class="<?php echo esc_attr( $field['class'] ); ?>" <?php checked( (int) get_user_meta( $user->ID, $key, true ), 1, true ); ?> />
									<?php
								} elseif ( ! empty( $field['type'] ) && 'button' === $field['type'] ) {
									?>
									<button type="button" id="<?php echo esc_attr( $key ); ?>" class="button <?php echo esc_attr( $field['class'] ); ?>"><?php echo esc_html( $field['text'] ); ?></button>
									<?php
								} elseif ( ! empty( $field['type'] ) && 'textarea' === $field['type'] ) {
									?>
									<textarea id="<?php echo esc_attr( $key ); ?>" class="<?php echo ( ! empty( $field['class'] ) ? esc_attr( $field['class'] ) : 'regular-text' ); ?>" name="<?php echo esc_attr( $key ); ?>" rows="5" cols="40"><?php echo esc_textarea( $this->get_user_meta( $user->ID, $key ) ); ?></textarea>
									<?php
								} elseif ( ! empty( $field['type'] ) && 'image_url' === $field['type'] ) {
									$cdhl_user_avatar      = $this->get_user_meta( $user->ID, $key );
									$cdhl_user_avatar_path = $this->get_user_meta( $user->ID, "{$key}_path" );
									$url                   = $cdhl_user_avatar ? $cdhl_user_avatar : '';
									$path                  = $cdhl_user_avatar_path ? $cdhl_user_avatar_path : '';
									// var_dump( $url );
									// var_dump( $path );

									$file_selector_label = $cdhl_user_avatar ? basename( $cdhl_user_avatar ) : '';
									$user_img_default    = trailingslashit( CDHL_URL ) . 'images/img-placeholder.jpg';
									$allowed_mimetypes   = array();

									foreach ( $field['allowed_file_types'] as $extension ) {
										$mimetype = mime_content_type( trailingslashit( CDHL_PATH ) . "images/placeholder-image.{$extension}" );
										if ( ! in_array( $mimetype, $allowed_mimetypes ) ) {
											$allowed_mimetypes[] = $mimetype;
										}
									}

									$field['allowed_mimetypes']      = $allowed_mimetypes;
									$field['default_image']          = $user_img_default;
									$field['invalid_file_header']    = esc_html__( 'Invalid file', 'cardealer-helper' );
									$field['invalid_file_message']   = esc_html__( 'The selected file is not an allowed file type. Please select another file.', 'cardealer-helper' );
									$field['no_file_selected_label'] = esc_html__( 'No file selected', 'cardealer-helper' );

									$allowed_file_types = array();

									foreach ( $field['allowed_file_types'] as $allowed_file_type ) {
										if ( ! in_array( $allowed_file_type, $allowed_file_types ) ) {
											$allowed_file_types[] = strtoupper( $allowed_file_type );
										}
									}

									$file_note = sprintf(
										/* translators: %1$s: allowed file types, %2$s: image dimension */
										esc_html__( '%1$s (Dimension: %2$s or above)', 'cardealer-helper' ),
										implode( ', ', $allowed_file_types ),
										sprintf( '%1$sx%2$s', $field['image_crop']['width'], $field['image_crop']['height'] )
									);
									$field['image_size_error'] = sprintf(
										/* translators: %s: image dimension */
										esc_html__( 'The selected image size must be %s or above.', 'cardealer-helper' ),
										sprintf( '%1$sx%2$s', $field['image_crop']['width'], $field['image_crop']['height'] )
									);
									?>
									<div class="cdhl-profile-field cdhl-profile-field-type-image_url" data-field_key="<?php echo esc_attr( $key ); ?>" data-field_data="<?php echo esc_attr( json_encode( $field ) ); ?>" data-file_selector_label="<?php echo esc_attr( $file_selector_label ); ?>">
										<div class="image-url-preview">
											<div id="<?php echo esc_attr( $key ); ?>-preview" class="image-url-preview-el <?php echo esc_attr( $cdhl_user_avatar ? 'with-image' : 'without-image' ); ?>">
												<img class="image-url-preview-img img-responsive img-thumb" src="<?php echo esc_url( $cdhl_user_avatar ? $cdhl_user_avatar : $user_img_default ); ?>"/>
											</div>
										</div>
										<div class="image-url-content">
											<div class="image-url-upload">
												<a href="#" class="button button-primary select-image"><?php echo esc_html( $field['file_btn'] ); ?></a>
												<div class="select-file-info">
													<div class="select-file-label"><?php esc_html_e( 'No file selected', 'cardealer-helper' ); ?></div>
													<div class="select-file-note"><?php echo esc_html( $file_note ); ?></div>
												</div>
												<input
													type="file"
													id="<?php echo esc_attr( $key ); ?>_selector"
													name="<?php echo esc_attr( $key ); ?>_selector"
													class="form-control image-url-selector"
													accept="image/png, image/jpeg"
													multiple="false"
												/>
											</div>
										</div>
										<?php
										$field_class = 'hidden';
										$field_type  = 'hidden';
										if ( defined('PGS_DEV_DEBUG') && PGS_DEV_DEBUG ) {
											$field_type  = 'text';
											$field_class = '';
										}
										?>
										<div class="image-url-hidden <?php echo esc_attr( $field_class ); ?>">
											<input type="<?php echo esc_attr( $field_type ); ?>" class="form-control regular-full" name="<?php echo esc_attr( $key ); ?>" id="<?php echo esc_attr( $key ); ?>_url" value="<?php echo esc_attr( $url ); ?>" />
											<input type="<?php echo esc_attr( $field_type ); ?>" class="form-control regular-full" name="<?php echo esc_attr( $key ); ?>_path" id="<?php echo esc_attr( $key ); ?>_path" value="<?php echo esc_attr( $path ); ?>" />
										</div>
									</div>
									<?php
								} elseif ( ! empty( $field['type'] ) && 'google_map' === $field['type'] ) {
									wp_enqueue_script( 'cardealer-google-maps-apis' );
									wp_enqueue_script( 'cdhl-google-location-picker' );
									$address = '';
									$lat     = '40.712775';
									$lng     = '-74.005973';
									$zoom    = '10';
									if ( isset( $user->ID ) ) {
										$dealer_location = $this->get_user_meta( $user->ID, 'dealer_location' );
										if ( ! empty( $dealer_location ) ) {
											$address = $dealer_location['address'];
											$lng     = $dealer_location['lng'];
											$lat     = $dealer_location['lat'];
											$zoom    = $dealer_location['zoom'];
										}
									} else {
										// get the default.
										global $car_dealer_options;
										$location_exits = true;
										$lat            = '';
										$lan            = '';

										if (
											( isset( $car_dealer_options['default_value_lat'] ) && ! empty( $car_dealer_options['default_value_lat'] ) )
											&& ( isset( $car_dealer_options['default_value_long'] ) && ! empty( $car_dealer_options['default_value_long'] ) )
											&& ( isset( $car_dealer_options['default_value_zoom'] ) && ! empty( $car_dealer_options['default_value_zoom'] ) )
										) {
											$lat  = $car_dealer_options['default_value_lat'];
											$lng  = $car_dealer_options['default_value_long'];
											$zoom = $car_dealer_options['default_value_zoom'];
										}
									}
									if ( empty( $zoom ) ) {
										$zoom = 10;
									}
									?>
									<div class="dealer-location-wrapper" style="width:100%">
										<input
											id="<?php echo esc_attr( $key ); ?>"
											type="text"
											class="regular-text"
											data-name="<?php echo esc_attr( $key ); ?>"
											name="<?php echo esc_attr( $key ); ?>[address]"
											value="<?php echo esc_attr( $address ); ?>"
											placeholder="<?php echo esc_attr__( 'Enter your location', 'cardealer-helper' ); ?>"
											style="margin-bottom:15px;"
										/>
										<input type="hidden" class="form-control" name="<?php echo esc_attr( $key ); ?>[lat]" style="width: 110px" id="<?php echo esc_attr( $key ); ?>_lat" value="<?php echo esc_attr( $lat ); ?>" />
										<input type="hidden" class="form-control" name="<?php echo esc_attr( $key ); ?>[lng]" style="width: 110px" id="<?php echo esc_attr( $key ); ?>_lng" value="<?php echo esc_attr( $lng ); ?>" />
										<input type="hidden" class="form-control" name="<?php echo esc_attr( $key ); ?>[zoom]" style="width: 110px" id="<?php echo esc_attr( $key ); ?>_zoom" value="<?php echo esc_attr( $zoom ); ?>" />
										<div id="dealer-location-map" style="width:100%;height:300px;border:1px solid #aaa;"></div>
									</div>
									<?php
								} else {
									?>
									<input type="text" name="<?php echo esc_attr( $key ); ?>" id="<?php echo esc_attr( $key ); ?>" value="<?php echo esc_attr( $this->get_user_meta( $user->ID, $key ) ); ?>" class="<?php echo ( ! empty( $field['class'] ) ? esc_attr( $field['class'] ) : 'regular-text' ); ?>" />
									<?php
								}
								?>
								<p class="description"><?php echo wp_kses_post( $field['description'] ); ?></p>
							</td>
						</tr>
						<?php
					}
					?>
				</table>
				<?php
			}
		}

		/**
		 * Save Address Fields on edit user pages.
		 *
		 * @param int $user_id User ID of the user being saved
		 */
		public function save_meta_fields( $user_id ) {
			// Check that the current user have the capability to edit the $user_id
			if ( ! apply_filters( 'cardealer_current_user_can_edit_user_profile_meta_fields', current_user_can( 'edit_user' ), $user_id ) ) {
				return;
			}

			$fieldsets = $this->get_fields();

			foreach ( $fieldsets as $fieldset_key => $fieldset ) {

				// Skip, if fields are empty.
				if ( empty( $fieldset['fields'] ) ) {
					continue;
				}

				$fields = $fieldset['fields'];

				// foreach ( $fields as $key => $field ) {
				foreach ( $fields as $field_key => $field ) {

					if ( isset( $field['type'] ) ) {
						if ( 'checkbox' === $field['type'] ) {
							update_user_meta( $user_id, $field_key, isset( $_POST[ $field_key ] ) );
						}elseif ( 'image_url' === $field['type'] ) {
							//continue;
							// $this->images_handler( $key, $field, $user_id, $_FILES );
							update_user_meta( $user_id, $field_key, $_POST[ $field_key ] );
							update_user_meta( $user_id, "{$field_key}_path", $_POST[ "{$field_key}_path" ] );
						} else {
							update_user_meta( $user_id, $field_key, cdhl_clean( $_POST[ $field_key ] ) );
						}
					} elseif ( isset( $_POST[ $field_key ] ) ) {
						update_user_meta( $user_id, $field_key, cdhl_clean( $_POST[ $field_key ] ) );
					}
				}
			}
		}

		function social_profile_fields( $fields ) {

			$fields['social_profiles'] = array(
				'title'  => esc_html__( 'Social Profiles', 'cardealer-helper' ),
				'fields' => apply_filters( 'cardealer_user_profile_meta_fields_social_profiles', array() ),
			);

			return $fields;
		}

		function social_profiles( $fields ) {

			$fields = array_merge(
				$fields,
				array(
					'facebook' => array(
						'label'       => __( 'Facebook', 'cardealer-helper' ),
						'description' => '',
						'icon'        => 'fab fa-facebook-f',
					),
					'twitter'  => array(
						'label'       => __( 'Twitter', 'cardealer-helper' ),
						'description' => '',
						'icon'        => 'fab fa-twitter',
					),
					'linkedin'    => array(
						'label'       => __( 'LinkedIn', 'cardealer-helper' ),
						'description' => '',
						'icon'        => 'fab fa-linkedin-in',
					),
					'pinterest'    => array(
						'label'       => __( 'Pinterest', 'cardealer-helper' ),
						'description' => '',
						'icon'        => 'fab fa-pinterest-p',
					),
					'instagram'    => array(
						'label'       => __( 'Instagram', 'cardealer-helper' ),
						'description' => '',
						'icon'        => 'fab fa-instagram',
					),
				)
			);

			return $fields;
		}

		/**
		 * Get Address Fields for the edit user pages.
		 *
		 * @return array Fields to display which are filtered through cardealer_user_profile_meta_fields before being returned
		 */
		public function get_fields() {

			$fields = apply_filters( 'cardealer_user_profile_meta_fields', array() );

			return $fields;
		}

		/**
		 * Get user meta for a given key, with fallbacks to core user info for pre-existing fields.
		 *
		 * @since 3.1.0
		 * @param int    $user_id User ID of the user being edited
		 * @param string $key     Key for user meta field
		 * @return string
		 */
		protected function get_user_meta( $user_id, $key ) {
			$value = get_user_meta( $user_id, $key, true );

			/*
			$existing_fields = array( 'billing_first_name', 'billing_last_name' );
			if ( ! $value && in_array( $key, $existing_fields ) ) {
				$value = get_user_meta( $user_id, str_replace( 'billing_', '', $key ), true );
			} elseif ( ! $value && ( 'billing_email' === $key ) ) {
				$user  = get_userdata( $user_id );
				$value = $user->user_email;
			}
			*/

			return $value;
		}

		public function profile_image_upload_callback() {
			$nonce_value = isset( $_POST['security'] ) ? wp_unslash( $_POST['security'] ) : '';
			$result      = array();

			if ( empty( $nonce_value ) || ! wp_verify_nonce( $nonce_value, 'file_upload' ) ) {
				wp_send_json_error( array(
					'message' => esc_html__( 'Unable to verify security nonce. Please rry again later.', 'cardealer-helper' ),
				) );
			} else {
				$user_id = (int) ( isset( $_POST['user_id'] ) && ! empty( $_POST['user_id'] ) ? sanitize_text_field( wp_unslash( $_POST['user_id'] ) ) : '' );
				if ( empty( $user_id ) ) {
					wp_send_json_error( array(
						'message' => esc_html__( 'The User ID is not provided. Please enter the User ID.', 'cardealer-helper' ),
					) );
				} else {
					if ( ! empty( $_FILES ) ) {
						$fieldsets = $this->get_fields();
						$files     = $_FILES;

						foreach ( $fieldsets as $fieldset_k => $fieldset ) {
							if ( isset( $fieldset['fields'] ) && is_array( $fieldset['fields'] ) && ! empty( $fieldset['fields'] ) ) {
								$fields = $fieldset['fields'];
								foreach ( $fields as $field_k => $field ) {
									if ( 'image_url' === $field['type'] && isset( $files[ $field_k ] ) ) {
										$image_data = $this->images_handler( $field_k, $field, $user_id, $files );
										if( is_wp_error( $image_data ) ) {
											$result[ $field_k ] = array(
												'status'  => false,
												'message' => $image_data->get_error_message(),
												'metas'   => array(),
											);
										} else {
											$result[ $field_k ] = array(
												'status'  => true,
												'message' => esc_html__( 'Image uploaded successfully.', 'cardealer-helper' ),
												'metas'   => $image_data,
											);
										}
									}
								}
							}
						}
					}
					wp_send_json_success( $result );
				}
			}
		}

		function images_handler( $field_name, $image_field, $user_id, $files ) {
			$profile_images_update = array();
			$allowed_file_types    = array( 'jpg', 'jpeg', 'png', 'JPG', 'JPEG', 'PNG' );
			$result                = array();

			if ( isset( $image_field['allowed_file_types'] ) && is_array( $image_field['allowed_file_types'] ) && ! empty( $image_field['allowed_file_types'] ) ) {
				$allowed_file_types = $image_field['allowed_file_types'];
			}

			if (
				isset( $files[ $field_name ] )
				&& is_array( $files[ $field_name ] )
				&& ! empty( $files[ $field_name ] )
				&& isset( $files[ $field_name ]['name'] )
				&& ! empty( $files[ $field_name ]['name'] )
			) {
				$raw_image          = $files[ $field_name ];
				$raw_image_pathinfo = pathinfo( $raw_image['name'] );
				$raw_image_ext      = $raw_image_pathinfo['extension'];

				if ( ! in_array( $raw_image_ext, $allowed_file_types ) ) {
					return new WP_Error( 'invalid_file_type', esc_html__( 'The %s file is not an allowed file type. Please select another file.', 'cardealer-helper' ) );
				} else {
					// Upload full image.
					if ( ! function_exists( 'wp_handle_upload' ) ) {
						require_once ABSPATH . 'wp-admin/includes/file.php';
					}

					// Register custom upload path.
					add_filter( 'upload_dir', array( $this, 'profile_image_upload_dir' ) );

					$upload_dir  = wp_upload_dir();
					$upload_url  = $upload_dir['url'];
					$upload_path = $upload_dir['path'];

					// Upload file.
					$uploaded_image = wp_handle_upload( $raw_image, array( 'test_form' => false ) );

					// Unregister custom upload path.
					remove_filter( 'upload_dir', array( $this, 'profile_image_upload_dir' ) );

					if ( is_wp_error( $uploaded_image ) ) {
						return $uploaded_image;
					} else {
						// Crop image.
						$uploaded_image_cropped = image_make_intermediate_size(
							$uploaded_image['file'],
							$image_field['image_crop']['width'],
							$image_field['image_crop']['height'],
							$image_field['image_crop']['crop']
						);

						// Delete original uploaded image.
						if ( file_exists( $uploaded_image['file'] ) ) {
							unlink( $uploaded_image['file'] );
						}

						if ( ! $uploaded_image_cropped ) {
							return new WP_Error( 'crop_error', esc_html__( 'Unable to crop image. Please try again.', 'cardealer-helper' ) );
						} else {
							// Get path and url of cropped image.
							$uploaded_image_cropped_url  = $upload_url . '/' . $uploaded_image_cropped['file'];
							$uploaded_image_cropped_path = $upload_path . '/' . $uploaded_image_cropped['file'];
							$uploaded_image_cropped_path = str_replace( array('/', '\\'), '/', $uploaded_image_cropped_path );

							// Delete old avatar.
							$old_avatar_path = get_the_author_meta( "{$field_name}_path", $user_id );

							if (
								! empty( $old_avatar_path )
								&& $uploaded_image_cropped_path !== $old_avatar_path
								&& file_exists( $old_avatar_path )
							) {

								/*Check if prev avatar exists in another users except current user*/
								$args     = array(
									'meta_key'     => "{$field_name}_path",
									'meta_value'   => $old_avatar_path,
									'meta_compare' => '=',
									'exclude'      => array( $user_id ),
								);
								$users_db = get_users( $args );
								if ( empty( $users_db ) ) {
									unlink( $old_avatar_path );
								}
							}

							$result = array(
								'url'  => $uploaded_image_cropped_url,
								"path" => $uploaded_image_cropped_path,
							);
						}
					}
				}
			}
			return $result;
		}

		/**
		 * Override the default upload path.
		 *
		 * @param   array   $dir
		 * @return  array
		 */
		function profile_image_upload_dir( $dir ) {
			return array(
				'path'   => $dir['basedir'] . '/dealer-profile-images',
				'url'    => $dir['baseurl'] . '/dealer-profile-images',
				'subdir' => '/dealer-profile-images',
			) + $dir;
		}
	}

endif;

new CarDealer_Admin_Profile();
