<?php
use Elementor\Controls_Stack;
use Elementor\Plugin;
use Elementor\Utils;

/**
 * Register Post Type for Templates.
 *
 * @author  TeamWP @Potenza Global Solutions
 * @package car-dealer-helper
 */

/**
 * CDHL_CPT_Template class.
 */
class CDHL_CPT_Template {

	/**
	 * Post type.
	 */
	const CPT = 'cardealer_template';

	/**
	 * Template Type.
	 */
	const TEMPLATE_TYPE_META = 'cardealer_template_type';

	/**
	 * Template Category.
	 */
	const TEMPLATE_CATEGORY = 'cardealer_template_category';

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'register_cpt' ) );
		add_action( 'admin_bar_menu', array( $this, 'remove_adminbar_link' ), 9999 );
		add_filter( 'in_admin_header', array( $this, 'action_buttons' ) );
		add_filter( 'views_edit-' . self::CPT, array( $this, 'display_notice' ) );
		add_filter( 'views_edit-' . self::CPT, array( $this, 'display_tabs' ) );
		add_filter( 'pre_get_posts', array( $this, 'filter_posts' ) );
		add_filter( 'display_post_states', array( $this, 'post_state' ), 10, 2 );
		add_action( 'admin_footer', array( $this, 'new_template_popup' ) );
		add_filter( 'cardealer_template_types_data', array( $this, 'cardealer_template_types_data' ), 20 );
		add_filter( 'body_class', array( $this, 'body_classes' ) );
		add_filter( 'cardealer_hide_header_banner', array( $this, 'hide_header_banner' ), 20 );

		// Admin Actions.
		add_action( 'admin_action_cardealer_templates_new_post', array( $this, 'create_new_template' ) );
		add_action( 'init', array( $this, 'wpbakery_set_support' ) );

		// Add template type meta box.
		add_action( 'add_meta_boxes_' . self::CPT, array( $this, 'meta_boxes' ) );
		add_action( 'save_post', array( $this, 'save_postdata' ), 10, 3 );
	}

	/**
	 * Register CPT function.
	 *
	 * @return void
	 */
	public function register_cpt() {
		$cpt_admin_cols = array(   // Add some custom columns to the admin screen.
			self::TEMPLATE_TYPE_META => array(
				'title'    => esc_html__( 'Type', 'cardealer-helper' ),
				'function' => array( $this, 'template_type_admin_column_value' ),
			),
		);

		if ( apply_filters( 'cardealer_templates_enable_category', false ) ) {
			$cpt_admin_cols[ self::TEMPLATE_CATEGORY ] = array(
				'title'    => esc_html__( 'Category', 'cardealer-helper' ),
				'taxonomy' => self::TEMPLATE_CATEGORY,
			);
		}

		$cpt_admin_cols[] = 'date';

		register_extended_post_type(
			self::CPT,
			array(
				'show_in_feed' => false,
				'show_in_rest' => false,
				'show_in_menu' => ( apply_filters( 'cardealer_templates_show_in_menu', 'edit.php?post_type=cars' ) ),
				'archive'      => array( // Show all posts on the post type archive.
					'nopaging' => true,
				),
				'admin_cols'   => $cpt_admin_cols,
				'labels'       => array(
					'all_items' => esc_html__( 'Templates', 'cardealer-helper' ),
				),
				'supports'     => array( 'title', 'editor', 'thumbnail', 'author', 'elementor' ),
				'rewrite'      => false,
				'hierarchical' => false,
			),
			// Override the base names used for labels.
			array(
				'singular' => esc_html__( 'Template', 'cardealer-helper' ),
				'plural'   => esc_html__( 'Templates', 'cardealer-helper' ),
				'slug'     => 'templates',
			)
		);

		if ( apply_filters( 'cardealer_templates_enable_category', false ) ) {
			register_extended_taxonomy(
				self::TEMPLATE_CATEGORY,
				array( self::CPT ),
				array(
					'meta_box'         => 'radio', // Use radio buttons in the meta box for this taxonomy on the post editing screen.
					'dashboard_glance' => false,   // Show this taxonomy in the 'At a Glance' dashboard widget.
					'admin_cols'       => array(   // Add a custom column to the admin screen.
						'updated' => array(
							'title'       => esc_html__( 'Updated', 'cardealer-helper' ),
							'meta_key'    => 'updated_date', // phpcs:ignore WordPress.DB.SlowDBQuery
							'date_format' => 'd/m/Y',
						),
					),
					'hierarchical'     => false,
					'exclusive'        => true,
					'allow_hierarchy'  => false,
					'required'         => true,
				),
				// Override the base names used for labels.
				array(
					'singular' => 'Template Category',
					'plural'   => 'Template Categories',
					'slug'     => 'cardealer-template-category',
				)
			);
		}
	}

	function meta_boxes() {
		if ( apply_filters( 'cardealer_templates_allow_template_settings_metabox', false ) ) {
			add_meta_box(
				'cardealer_templates__template_settings_metabox',
				esc_html__( 'Template Settings', 'cardealer-helper' ),
				array( $this, 'template_settings_metabox_callback' ),
				array(
					self::CPT
				),
				'side',
				'high'
			);
		}
	}

	function template_settings_metabox_callback( $post ) {
		$value          = get_post_meta( $post->ID, self::TEMPLATE_TYPE_META, true );
		$template_types = $this->template_types( 'id=>label' );
		?>
		<label for="cardealer_template_type"><?php echo esc_html__( 'Template Type', 'cardealer-helper' ); ?></label>
		<select name="cardealer_template_type" id="cardealer_template_type" class="postbox">
			<option value=""><?php echo esc_html__( 'Select Template Type', 'cardealer-helper' ); ?></option>
			<?php
			foreach ( $template_types as $template_type_k => $template_type_name ) {
				?>
				<option value="<?php echo esc_attr( $template_type_k ); ?>" <?php selected( $value, $template_type_k ); ?>><?php echo esc_html( $template_type_name ); ?></option>
				<?php
			}
			?>
		</select>
		<?php
	}

	/**
     * Save the meta box selections.
     *
	 * @param int     $post_id Post ID.
	 * @param WP_Post $post    Post object.
	 * @param bool    $update  Whether this is an existing post being updated.
     */
    function save_postdata( int $post_id, $post, $update ) {
        if ( self::CPT === $post->post_type && $update && array_key_exists( 'cardealer_template_type', $_POST ) ) {
            update_post_meta( $post_id, 'cardealer_template_type', sanitize_text_field( wp_unslash( $_POST['cardealer_template_type'] ) ) );
        }
    }

	/**
	 * Remove add template link from admin bar.
	 *
	 * @return void
	 */
	public function remove_adminbar_link() {
		global $wp_admin_bar;
		$wp_admin_bar->remove_node( 'new-' . self::CPT );
	}

	/**
	 * Set CPT support in the WPBakery role manager.
	 *
	 * @return void
	 */
	public function wpbakery_set_support() {

		$cd_wpb_set_roles = get_option( 'cd_wpb_template_set' );
		if ( class_exists( 'WPBakeryVisualComposerAbstract' ) && ! $cd_wpb_set_roles ) {

			$file = vc_path_dir( 'SETTINGS_DIR', 'class-vc-roles.php' );
			if ( ! file_exists( $file ) ) {
				return;
			}

			require_once $file;

			if ( ! class_exists( 'Vc_Roles' ) ) {
				return;
			}

			$vc_roles = new Vc_Roles();

			$get_posts = array();
			$roles     = vc_user_roles_get_all();

			if ( isset( $roles['post_types']['capabilities'] ) && $roles['post_types']['capabilities'] ) {
				$get_posts = $roles['post_types']['capabilities'];
			}

			$get_posts['_state']             = 'custom';
			$get_posts['page']               = 1;
			$get_posts['cardealer_template'] = 1;

			$vc_roles->save(
				array(
					'administrator' => array(
						'post_types' => $get_posts,
					),
				)
			);

			update_option( 'cd_wpb_template_set', true );
		}
	}

	/**
	 * Template type admin column value.
	 *
	 * @param WP_Post $post Post object.
	 * @return void
	 */
	public function template_type_admin_column_value( $post ) {
		$template_type_label = '';
		$meta_value          = get_post_meta( $post->ID, self::TEMPLATE_TYPE_META, true );
		$template_types      = $this->template_types( 'id=>label' );

		if ( isset( $template_types[ $meta_value ] ) && ! empty( $template_types[ $meta_value ] ) ) {
			$template_type_label = $template_types[ $meta_value ];
		}

		if ( ! empty( $template_type_label ) ) {
			$baseurl = add_query_arg(
				array(
					'post_type'              => self::CPT,
					self::TEMPLATE_TYPE_META => $meta_value,
				),
				admin_url( 'edit.php' )
			);
			echo wp_kses(
				sprintf(
					'<a href="%s">%s</a>',
					esc_url( $baseurl ),
					esc_html( $template_type_label )
				),
				array(
					'a' => array(
						'href' => true,
					),
				)
			);
		} else {
			echo '&mdash;';
		}

	}

	/**
	 * Display action buttons.
	 *
	 * @return void
	 */
	public function action_buttons() {
		$current_screen = get_current_screen();
		if ( in_array( $current_screen->id, array( 'edit-cardealer_template', 'cardealer_template' ), true ) ) {
			?>
			<div class="cardealer_templates-actions" style="display:none;">
				<a class="cardealer_templates-action cardealer_templates-action-new" href="#"><?php echo esc_html__( 'Add New', 'cardealer-helper' ); ?></a>
				<a class="cardealer_templates-action cardealer_templates-action-document" href="https://docs.potenzaglobalsolutions.com/docs/cardealer/?section=cardealer-templates" target="_blank"><?php echo esc_html__( 'Document', 'cardealer-helper' ); ?></a>
			</div>
			<?php
		}
	}

	/**
	 * Template type popup contents.
	 *
	 * @param string $data The data to print.
	 * @return string
	 */
	public function new_template_popup( $data ) {

		if ( ! self::is_current_screen() ) {
			return $data;
		}

		$template_types       = $this->template_types( 'id=>label' );
		$active_builders      = cardealer_get_active_page_builders();
		$predefined_templates = array_values( cardealer_template()->templates );
		?>
		<!-- Modal -->
		<div class="cardealer-bootstrap">
			<div id="cdtmpl-modal" class="modal fade cdtmpl-modal" tabindex="-1" aria-labelledby="cdtmpl-modal-label" aria-hidden="true">
				<div class="modal-dialog modal-dialog-centered modal-dialog-scrollable modal-xl">
					<div class="modal-content">
						<div class="modal-header">
							<h5 class="modal-title" id="cdtmpl-modal-label"><?php echo esc_html__( 'New Template', 'cardealer-helper' ); ?></h5>
							<button id="cdtmpl-modal-close-icon" type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
						</div>
						<div class="modal-body">
							<form id="cdtmpl-modal-form" action="<?php esc_url( admin_url( '/edit.php' ) ); ?>">
								<?php
								if ( count( $active_builders ) > 1 && ! apply_filters( 'cardealer_allow_multiple_page_builder', false ) ) {
									?>
									<div class="alert alert-danger d-flex" role="alert">
										<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" fill="currentColor" class="bi flex-shrink-0 me-2" viewBox="0 0 16 16"><path d="M8.982 1.566a1.13 1.13 0 0 0-1.96 0L.165 13.233c-.457.778.091 1.767.98 1.767h13.713c.889 0 1.438-.99.98-1.767L8.982 1.566zM8 5c.535 0 .954.462.9.995l-.35 3.507a.552.552 0 0 1-1.1 0L7.1 5.995A.905.905 0 0 1 8 5zm.002 6a1 1 0 1 1 0 2 1 1 0 0 1 0-2z"></path></svg></svg>
										<div class="cdtmpl-alert-content">
											<p><?php echo esc_html__( 'More than one page-builders are active. Please keep only one page-builder active and deactivate all other page builders.', 'cardealer-helper' ); ?></p>
										</div>
									</div>
									<?php
								} else {
									?>
									<input type="hidden" name="post_type" value="<?php echo esc_attr( self::CPT ); ?>">
									<input type="hidden" name="action" value="cardealer_templates_new_post">
									<input type="hidden" name="_wpnonce" value="<?php echo wp_create_nonce( 'cardealer_templates_action_new_post' ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>">
									<?php
									if ( count( $active_builders ) > 1 && apply_filters( 'cardealer/cardealer-templates/allow_builder_select', false ) ) {
										$page_builders = cardealer_get_page_builders();
										?>
										<div class="row mb-3">
											<label for="cdtmpl-builder-type" class="col-sm-3 col-form-label"><?php echo esc_html__( 'Page Builder', 'cardealer-helper' ); ?></label>
											<div class="col-sm-9">
												<select id="cdtmpl-builder-type" class="form-select" name="builder_type" required>
													<option value=""><?php echo esc_html__( 'Select...', 'cardealer-helper' ); ?></option>
													<?php
													foreach ( $active_builders as $active_builder ) {
														printf(
															'<option value="%1$s">%2$s</option>',
															esc_attr( $active_builder ),
															esc_html( $page_builders[ $active_builder ]['label'] )
														);
													}
													?>
												</select>
											</div>
										</div>
										<?php
									} else {
										$active_builder_key   = array_key_first( $active_builders );
										$active_builder_value = $active_builders[ $active_builder_key ];
										?>
										<input type="hidden" id="cdtmpl-builder-type" name="builder_type" value="<?php echo esc_attr( $active_builder_value ); ?>">
										<?php
									}
									?>
									<div class="row mb-3">
										<label for="cdtmpl-template-type" class="col-sm-3 col-form-label"><?php echo esc_html__( 'Template Type', 'cardealer-helper' ); ?></label>
										<div class="col-sm-9">
											<?php
											if ( count( $template_types ) === 1 ) {
												$template_type_key   = array_key_first( $template_types );
												$template_type_value = $template_types[ $template_type_key ];
												?>
												<input type="text" readonly class="form-control-plaintext" value="<?php echo esc_attr( $template_type_value ); ?>">
												<input type="hidden" name="template_type" value="<?php echo esc_attr( $template_type_key ); ?>">
												<?php
											} else {
												?>
												<select id="cdtmpl-template-type" class="form-select" name="template_type" required>
													<option value=""><?php echo esc_html__( 'Select...', 'cardealer-helper' ); ?></option>
													<?php
													foreach ( $template_types as $value => $type_title ) {
														printf( '<option value="%1$s">%2$s</option>', esc_attr( $value ), esc_html( $type_title ) );
													}
													?>
												</select>
												<?php
											}
											?>
										</div>
									</div>
									<div class="row mb-3">
										<label for="cdtmpl-post-title" class="col-sm-3 col-form-label"><?php echo esc_html__( 'Template Name', 'cardealer-helper' ); ?></label>
										<div class="col-sm-9">
											<input type="text" id="cdtmpl-post-title" class="form-control" name="post_title" placeholder="<?php echo esc_attr__( 'Enter template name (optional)', 'cardealer-helper' ); ?>">
											<small><em><?php echo esc_attr__( 'If the title is empty, it will be created automatically.', 'cardealer-helper' ); ?></em></small>
										</div>
									</div>
									<?php
									/**
									 * Template library dialog fields.
									 *
									 * Fires after CarDealer new template dialog fields are displayed.
									 */
									do_action( 'cardealer/cardealer-templates/create_new_dialog_fields' );

									// Radio check referrence: https://codepen.io/munsheerek/pen/QmEWwq
									?>
									<div class="row predefined-templates-wrapper">
										<div class="predefined-templates-header col-sm-12 mb-3">
											<h6 class="predefined-templates-label" style="margin: 0;"><?php echo esc_html__( 'Predefined Templates (Optional)', 'cardealer-helper' ); ?></h6>
											<em><small><?php echo esc_html__( 'If a predefined template is not selected, the blank template will be created.', 'cardealer-helper' ); ?></small> <small class="text-danger"><?php echo esc_html__( 'Click a template to select/unselect.', 'cardealer-helper' ); ?></small></em>
										</div>
										<div class="predefined-templates-content col-sm-12">
											<div class="spinner-border" role="status">
												<span class="visually-hidden"><?php echo esc_html__( 'Loading...', 'cardealer-helper' ); ?></span>
											</div>
										</div>
									</div>
									<?php
								}
								?>
							</form>
						</div>
						<div class="modal-footer">
							<button id="cdtmpl-close-button" type="button" class="btn btn-secondary" data-bs-dismiss="modal"><?php echo esc_html__( 'Close', 'cardealer-helper' ); ?></button>
							<?php
							if ( 1 === count( $active_builders ) || apply_filters( 'cardealer_allow_multiple_page_builder', false ) ) {
								?>
								<button id="cdtmpl-create-template-button" class="btn btn-primary" form="cdtmpl-modal-form"><?php echo esc_html__( 'Create Template', 'cardealer-helper' ); ?></button>
								<?php
							}
							?>
						</div>
					</div>
				</div>
			</div>
		</div>
		<script type="text/javascript">var cardealer_predefined_templates = <?php echo json_encode( $predefined_templates ); ?></script>
		<script type="text/template" id="tmpl-cardealer_templates--predefined-templates">
			<#
			if ( data.templates ) {
				#>
				<div class="row predefined-templates">
					<#
					_.each( data.templates, function( template_data, index ) {
						#>
						<div class="col-sm-4 predefined-template mb-3">
							<input id="{{{ template_data.id }}}" class="d-none predefined-template-imgcheck" type="radio" name="predefined-template" value="{{{ template_data.id }}}">
							<label data-for="{{{ template_data.id }}}" class="predefined-template-label border border-secondary">
								<img class="img-fluid predefined-template-thumbnail" src="{{{ template_data.thumbnail_url }}}" alt="{{{ template_data.title }}}" width="640" height="450">
								<div class="predefined-template-title bg-white p-2 px-3">{{{ template_data.title }}}</div>
								<div class="predefined-template-check-container"><div class="predefined-template-check"><i class="fa fa-check"></i></div></div>
							</label>
						</div>
						<#
					} );
					#>
				</div>
				<#
			}
			#>
		</script>
		<script type="text/template" id="tmpl-cardealer_templates--predefined-template--not-found">
			<div class="alert alert-info d-flex" role="alert">
				<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" fill="currentColor" class="bi flex-shrink-0 me-2" viewBox="0 0 16 16"><path d="M8.982 1.566a1.13 1.13 0 0 0-1.96 0L.165 13.233c-.457.778.091 1.767.98 1.767h13.713c.889 0 1.438-.99.98-1.767L8.982 1.566zM8 5c.535 0 .954.462.9.995l-.35 3.507a.552.552 0 0 1-1.1 0L7.1 5.995A.905.905 0 0 1 8 5zm.002 6a1 1 0 1 1 0 2 1 1 0 0 1 0-2z"></path></svg></svg>
				<div class="cdtmpl-alert-content">
					<?php echo esc_html__( 'No templates found.', 'cardealer-helper' ); ?>
				</div>
			</div>
		</script>
		<script type="text/template" id="tmpl-cardealer_templates--predefined-template--select-option">
			<div class="alert alert-info d-flex" role="alert">
				<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" fill="currentColor" class="bi flex-shrink-0 me-2" viewBox="0 0 16 16"><path d="M8.982 1.566a1.13 1.13 0 0 0-1.96 0L.165 13.233c-.457.778.091 1.767.98 1.767h13.713c.889 0 1.438-.99.98-1.767L8.982 1.566zM8 5c.535 0 .954.462.9.995l-.35 3.507a.552.552 0 0 1-1.1 0L7.1 5.995A.905.905 0 0 1 8 5zm.002 6a1 1 0 1 1 0 2 1 1 0 0 1 0-2z"></path></svg></svg>
				<div class="cdtmpl-alert-content">
					<p><?php echo esc_html__( 'Select template properties to display templates.', 'cardealer-helper' ); ?></p>
				</div>
			</div>
		</script>
		<script type="text/template" id="tmpl-cardealer_templates--predefined-template--loader">
			<div class="spinner-border" role="status"><span class="visually-hidden"><?php echo esc_html__( 'Loading...', 'cardealer-helper' ); ?></span></div>
		</script>
		<?php
		return $data;
	}

	/**
	 * Display multiple page-builder notice.
	 *
	 * @access public
	 *
	 * @param array $views An array of available list table views.
	 *
	 * @return array An updated array of available list table views.
	 */
	public function display_notice( $views ) {
		?>
		<div class="notice notice-error inline" style="padding: 15px 25px">
			<h2 style="margin-top: 0px;margin-bottom: 10px;"><?php echo esc_html__( 'Important Note', 'cardealer-helper' ); ?></h2>
			<ul style="font-size: 16px;padding-left: 15px;list-style: disc;margin-bottom: 0;">
				<li><?php echo esc_html__( 'If more than one page-builder is installed and activated, conflict may arise with some functionalities. So please just keep one active and deactivate the other builders.', 'cardealer-helper' ); ?></li>
				<li><?php echo esc_html__( 'When editing/previewing a template, the data displayed is FOR PREVIEW PURPOSES ONLY and to show how it will look on the front. Some content/elements may look or work differently as they should. For actual features view the actual post/page on the front where the template is used (like for Vehicle Details template view cars detail page).', 'cardealer-helper' ); ?></li>
			</ul>
		</div>
		<?php
		return $views;
	}

	/**
	 * Display template type tabs.
	 *
	 * Fired by `views_edit-elementor_library` filter.
	 *
	 * @access public
	 *
	 * @param array $views An array of available list table views.
	 *
	 * @return array An updated array of available list table views.
	 */
	public function display_tabs( $views ) {
		$current_type = $this->get_current_type();
		$active_class = ' nav-tab-active';

		if ( ! empty( $current_type ) ) {
			$active_class = '';
		}

		$baseurl = add_query_arg(
			array(
				'post_type' => self::CPT,
			),
			admin_url( 'edit.php' )
		);

		$template_types = $this->template_types( 'id=>label' );

		if ( 1 >= count( $template_types ) ) {
			return $views;
		}
		?>
		<div id="caldealer-elementor-template-tabs-wrapper" class="nav-tab-wrapper" style="margin-bottom: 15px;">
			<a class="nav-tab<?php echo esc_attr( $active_class ); ?>" href="<?php echo esc_url( $baseurl ); ?>"><?php echo esc_html__( 'All', 'cardealer-helper' ); ?></a>
			<?php
			foreach ( $template_types as $type => $type_label ) :
				$active_class = '';

				if ( $current_type === $type ) {
					$active_class = ' nav-tab-active';
				}
				$type_url = esc_url( add_query_arg( self::TEMPLATE_TYPE_META, $type, $baseurl ) );
				?>
				<a class="nav-tab<?php echo esc_attr( $active_class ); ?>" href="<?php echo esc_url( $type_url ); ?>"><?php echo esc_html( $type_label ); ?></a>
				<?php
			endforeach;
			?>
		</div>
		<?php
		return $views;
	}

	/**
	 * Get current template type.
	 *
	 * @return string
	 */
	public function get_current_type() {
		$current_type = '';
		if ( isset( $_REQUEST[ self::TEMPLATE_TYPE_META ] ) && ! empty( $_REQUEST[ self::TEMPLATE_TYPE_META ] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
			$current_type = sanitize_text_field( wp_unslash( $_REQUEST[ self::TEMPLATE_TYPE_META ] ) );             // phpcs:ignore WordPress.Security.NonceVerification
		}
		return $current_type;
	}

	/**
	 * Filter posts.
	 *
	 * @param WP_Query $query  The WP_Query instance.
	 * @return WP_Query
	 */
	public function filter_posts( $query ) {

		$current_type = $this->get_current_type();

		if ( is_admin() && $query->is_main_query() && self::CPT === $query->get( 'post_type' ) && ! empty( $current_type ) ) {

			$query->set(
				'meta_query',
				array(
					'relation' => 'OR',
					array(
						'key'   => self::TEMPLATE_TYPE_META,
						'value' => $current_type,
					),
				)
			);

		}

		return $query;
	}

	/**
	 * Add template built with post state.
	 *
	 * Fired by `display_post_states` filter.
	 *
	 * @access public
	 *
	 * @param array    $post_states An array of post display states.
	 * @param \WP_Post $post        The current post object.
	 *
	 * @return array A filtered array of post display states.
	 */
	public function post_state( $post_states, $post ) {

		if ( self::CPT === $post->post_type ) {
			$template_built_with = self::template_built_with( $post->ID );
			if ( $template_built_with && 'wpbakery' === $template_built_with ) {
				$post_states['wpbakery'] = esc_html__( 'WPBakery', 'cardealer-helper' );
			}
		}

		return $post_states;
	}

	/**
	 * Template types.
	 *
	 * @param string $fields Fields.
	 * @return array
	 */
	public static function template_types( $fields = '' ) {

		$template_types = array(
			'vehicle_detail' => array(
				'type'           => 'post_type',
				'page'           => 'single',
				'post_type'      => 'cars',
				'label'          => esc_html__( 'Vehicle Detail', 'cardealer-helper' ),
			),
			'vehicle_detail_mobile' => array(
				'type'      => 'post_type',
				'page'      => 'single',
				'post_type' => 'cars',
				'label'     => esc_html__( 'Vehicle Detail (Mobile)', 'cardealer-helper' ),
				'get_data_field' => 'vehicle_detail',
			),
		);

		$template_types = apply_filters( 'cardealer_template_types', $template_types );

		// Process template types.
		foreach ( $template_types as $template_type_k => $template_type_data ) {

			$template_type_data = array_merge(
				array(
					'id' => $template_type_k,
				),
				$template_type_data
			);

			if ( 'post_type' === $template_type_data['type'] && isset( $template_type_data['post_type'] ) && ! empty( $template_type_data['post_type'] ) ) {
				$post_type_object = get_post_type_object( $template_type_data['post_type'] );
				if ( $post_type_object ) {
					$template_type_data['post_type_object'] = $post_type_object;
				}
			}

			$template_types[ $template_type_k ] = $template_type_data;
		}

		if ( 'id=>label' === $fields ) {
			$template_types = array_column( $template_types, 'label', 'id' );
		}

		return $template_types;
	}

	/**
	 * Template type data.
	 *
	 * @return array
	 */
	public function cardealer_template_types_data() {
		return $this->template_types();
	}

	/**
	 * Add custom class in body.
	 *
	 * @param array $classes (array) the array to look within.
	 */
	public function body_classes( $classes ) {

		global $car_dealer_options, $post;

		if ( is_singular( 'cars' ) ) {
			$layout        = cardear_get_vehicle_detail_page_layout();
			$template_type = ( isset( $car_dealer_options['vehicle-template-type'] ) ) ? $car_dealer_options['vehicle-template-type'] : 'builder';

			if ( wp_is_mobile() ) {
				$hide_header = ( isset( $car_dealer_options['vehicle-builder-template-hide-header-mobile'] ) ) ? $car_dealer_options['vehicle-builder-template-hide-header-mobile'] : ( 'builder' === $template_type );
			}else {
				$hide_header = ( isset( $car_dealer_options['vehicle-builder-template-hide-header'] ) ) ? $car_dealer_options['vehicle-builder-template-hide-header'] : ( 'builder' === $template_type );
			}

			$hide_header = filter_var( $hide_header, FILTER_VALIDATE_BOOLEAN );

			if ( 'builder' === $layout && $hide_header ) {
				$classes[] = "single-cars-layout-{$layout}-hide-header";
			}
		}

		return $classes;
	}

	/**
	 * Hide page header in builder template.
	 *
	 * @param bool $hide_header_banner Hide page header.
	 */
	public function hide_header_banner( $hide_header_banner ) {
		global $car_dealer_options, $post;

		if ( is_singular( 'cars' ) ) {
			$layout        = cardear_get_vehicle_detail_page_layout();
			$template_type = ( isset( $car_dealer_options['vehicle-template-type'] ) ) ? $car_dealer_options['vehicle-template-type'] : 'builder';

			if ( wp_is_mobile() ) {
				$hide_header = ( isset( $car_dealer_options['vehicle-builder-template-hide-header-mobile'] ) ) ? $car_dealer_options['vehicle-builder-template-hide-header-mobile'] : ( 'builder' === $template_type );
			} else {
				$hide_header = ( isset( $car_dealer_options['vehicle-builder-template-hide-header'] ) ) ? $car_dealer_options['vehicle-builder-template-hide-header'] : ( 'builder' === $template_type );
			}

			$hide_header = filter_var( $hide_header, FILTER_VALIDATE_BOOLEAN );

			if ( 'builder' === $layout ) {
				$hide_header_banner = $hide_header;
			}
		}

		return $hide_header_banner;
	}

	/**
	 * Check is current screen.
	 *
	 * @return boolean
	 */
	private function is_current_screen() {
		global $pagenow, $typenow;

		return ( 'edit.php' === $pagenow || 'post.php' === $pagenow ) && self::CPT === $typenow;
	}

	/**
	 * Admin action new post.
	 *
	 * When a new post action is fired the title is set to 'Elementor' and the post ID.
	 *
	 * Fired by `admin_action_elementor_new_post` action.
	 *
	 * @since 1.9.0
	 * @access public
	 */
	public function create_new_template() {

		check_admin_referer( 'cardealer_templates_action_new_post' );

		$post_type_object = get_post_type_object( self::CPT );

		if ( ! current_user_can( $post_type_object->cap->edit_posts ) ) {
			return;
		}

		$template_types = $this->template_types( 'id=>label' );

		if ( empty( $_GET['template_type'] ) || ! array_key_exists( sanitize_text_field( wp_unslash( $_GET['template_type'] ) ), $template_types ) ) {
			return;
		}

		$post_data     = array();
		$template_type = sanitize_text_field( wp_unslash( $_GET['template_type'] ) );
		$page_builders = cardealer_get_page_builders();
		$builder_type  = ( isset( $_GET['builder_type'] ) && ! empty( $_GET['builder_type'] ) && array_key_exists( sanitize_text_field( wp_unslash( $_GET['builder_type'] ) ), $page_builders ) ) ? sanitize_text_field( wp_unslash( $_GET['builder_type'] ) ) : '';
		$post_title    = ( isset( $_GET['post_title'] ) && ! empty( $_GET['post_title'] ) ) ? sanitize_text_field( wp_unslash( $_GET['post_title'] ) ) : '';
		$update_title  = false;

		$predefined_templates = cardealer_template()->templates;
		$predefined_template  = ( isset( $_GET['predefined-template'] ) && ! empty( $_GET['predefined-template'] ) && array_key_exists( sanitize_text_field( wp_unslash( $_GET['predefined-template'] ) ), $predefined_templates ) ) ? sanitize_text_field( wp_unslash( $_GET['predefined-template'] ) ) : '';

		if ( empty( $post_title ) ) {
			$post_title   = esc_html__( 'Car Dealer Template', 'cardealer-helper' );
			$update_title = true;
		}

		$post_data['post_title'] = $post_title;
		$post_data['post_type']  = self::CPT;

		/**
		 * Filters the meta data of any new post created.
		 *
		 * @param array $meta Post meta data.
		 */
		$meta_data = apply_filters( 'cardealer/cardealer_templates/create_new_template/meta_data', array(), $post_data );

		if ( ! empty( $builder_type ) ) {
			$active_builder = $builder_type;
		} else {
			$active_builders = cardealer_get_active_page_builders();
			$active_builder  = ( count( $active_builders ) === 1 ) ? $active_builders[0] : '';
		}

		$meta_data[ self::TEMPLATE_TYPE_META ] = $template_type;

		$template_file = '';

		if ( ! empty( $predefined_template ) && ! empty( $predefined_templates[ $predefined_template ] ) ) {
			$template_data = $predefined_templates[ $predefined_template ];
			if ( isset( $template_data['template_file'] ) && file_exists( $template_data['template_file'] ) ) {
				$template_file = $template_data['template_file'];
			}
		}

		$elementor_import  = false;
		$elementor_content = false;

		if ( 'elementor' === $active_builder ) {
			$meta_data['_elementor_edit_mode'] = 'builder';
			if ( ! empty( $template_file ) ) {
				$data = json_decode( file_get_contents( $template_file ), true );
				if ( ! empty( $data ) ) {
					$content = $data['content'];
					if ( is_array( $content ) ) {
						$content = $this->process_export_import_content( $content, 'on_import' );
						$content = $this->replace_elements_ids( $content );

						$elementor_import  = true;
						$elementor_content = $content;
					}
				}
			}
		} elseif ( 'wpbakery' === $active_builder ) {
			$meta_data['_wpb_vc_js_status'] = 'true';

			if ( ! empty( $template_file ) ) {
				$template_content = require_once( $template_file );
				$template_content = trim( $template_content );

				if ( ! empty( $template_content ) ) {
					$post_data['post_content'] = $template_content;
				}
			}

		}

		$post_data['meta_input'] = $meta_data;

		/**
		 * Filters the meta data of any new post created.
		 *
		 * @param array $meta Post meta data.
		 */
		$post_data = apply_filters( 'cardealer/cardealer_templates/create_new_template/post_data', $post_data );

		// Insert post.
		$post_id = wp_insert_post( $post_data );

		if ( is_wp_error( $post_id ) ) {
			return;
		}

		if ( $elementor_import && $elementor_content ) {
			$document = Plugin::$instance->documents->get( $post_id );
			if ( $document ) {

				$editor_data = $document->get_elements_raw_data( $elementor_content );

				// We need the `wp_slash` in order to avoid the unslashing during the `update_post_meta`
				$json_value = wp_slash( wp_json_encode( $editor_data ) );

				// Don't use `update_post_meta` that can't handle `revision` post type
				update_post_meta( $post_id, '_elementor_data', $json_value );

			}
		}

		if ( $update_title ) {
			$post_data               = array();
			$post_data['ID']         = $post_id;
			$post_data['post_title'] = "$post_title #$post_id";

			wp_update_post( $post_data );
		}
		/*----------------------------------------------*/

		$redirect_action = ( 'elementor' === $active_builder ) ? 'elementor' : 'edit';
		$redirect_url    = add_query_arg(
			array(
				'post'   => $post_id,
				'action' => $redirect_action,
			),
			admin_url( 'post.php' )
		);

		wp_safe_redirect( $redirect_url );
		exit();
	}

	/**
	 * Get templates.
	 *
	 * @param string $template_type Template type.
	 * @return array
	 */
	public static function get_emplates( $template_type = 'vehicle_detail' ) {
		$templates = array();
		$args      = array(
			'post_type'           => self::CPT,
			'posts_per_page'      => -1,
			'orderby'             => 'date',
			'order'               => 'DESC',
			'suppress_filters'    => true,
			'ignore_sticky_posts' => true,
			'no_found_rows'       => true,
			'meta_query'          => array( // phpcs:ignore WordPress.DB.SlowDBQuery
				array(
					'key'   => self::TEMPLATE_TYPE_META,
					'value' => $template_type,
				),
			),
		);

		$get_posts = new WP_Query();
		$posts     = $get_posts->query( $args );
		$posts_kv  = array_column( $posts, 'post_title', 'ID' );
		if ( is_array( $posts_kv ) && ! empty( $posts_kv ) ) {
			$templates = $posts_kv;
		}
		return $templates;
	}

	/**
	 * Get template ID.
	 *
	 * @param string $template_type Template type.
	 * @return int
	 */
	public static function get_template_id( $template_type = 'vehicle_detail' ) {
		global $car_dealer_options, $post;

		$template_id = '';
		$opt_key     = "cardealer_template_option_$template_type";

		if ( isset( $car_dealer_options[ $opt_key ] ) && ! empty( $car_dealer_options[ $opt_key ] ) ) {
			$template_id_option = $car_dealer_options[ $opt_key ];

			$template_post = get_post( $template_id_option );
			if ( is_a( $template_post, 'WP_Post' ) && self::CPT === $template_post->post_type ) {
				$template_id = $template_post->ID;
			}
		}

		$meta_value = ( $post ) ? get_post_meta( $post->ID, "cardealer_template__{$template_type}", true ) : '';
		if ( $meta_value && ! empty( $meta_value ) ) {
			$template_id = $meta_value;
		}

		return (int) $template_id;
	}

	/**
	 * Get post ID.
	 *
	 * @param string $template_type Template type.
	 * @param array  $args          Query arguments.
	 * @return int
	 */
	public static function get_post_id( $template_type = 'vehicle_detail', $args = array() ) {
		global $car_dealer_options, $post;

		$template_types = self::template_types();
		$post_id        = 0;

		if ( $post && $template_types[ $template_type ]['post_type'] === $post->post_type ) {
			$post_id = $post->ID;
		} else {
			$opt_key = "cardealer_template_data_post_option_$template_type";

			if ( isset( $car_dealer_options[ $opt_key ] ) && ! empty( $car_dealer_options[ $opt_key ] ) ) {
				$post_id_option = $car_dealer_options[ $opt_key ];
				$post_data      = get_post( $post_id_option );

				if ( is_a( $post_data, 'WP_Post' ) && $template_types[ $template_type ]['post_type'] === $post_data->post_type ) {
					$post_id = $post_data->ID;
				}
			}

			if ( empty( $post_id ) ) {
				$posts = self::get_posts( $template_type, $args );
				$posts = array_column( $posts, 'post_title', 'ID' );

				if ( $posts && is_array( $posts ) && ! empty( $posts ) ) {
					foreach ( $posts as $post_id => $title_title ) {
						$post_id = $post_id;
						continue;
					}
				}
			}
		}
		return (int) $post_id;
	}

	/**
	 * Get posts.
	 *
	 * @param string $template_type  Template types.
	 * @param array  $args           Arguments.
	 * @return array
	 */
	public static function get_posts( $template_type = 'vehicle_detail', $args = array() ) {
		$posts = 0;

		$template_types = self::template_types();
		if (
			isset( $template_types[ $template_type ] ) && ! empty( $template_types[ $template_type ] )
			&& ( isset( $template_types[ $template_type ]['type'] ) && 'post_type' === $template_types[ $template_type ]['type'] )
			&& ( isset( $template_types[ $template_type ]['page'] ) && 'single' === $template_types[ $template_type ]['page'] )
			&& ( isset( $template_types[ $template_type ]['post_type'] ) && ! empty( $template_types[ $template_type ]['post_type'] ) && post_type_exists( $template_types[ $template_type ]['post_type'] ) )
		) {
			$defaults = array(
				'post_type'           => $template_types[ $template_type ]['post_type'],
				'numberposts'         => -1,
				'orderby'             => 'date',
				'order'               => 'ASC',
				'suppress_filters'    => true,
				'ignore_sticky_posts' => true,
				'no_found_rows'       => true,
			);

			$args      = wp_parse_args( $args, $defaults );
			$get_posts = new WP_Query();
			$posts     = $get_posts->query( $args );
		}

		return $posts;
	}

	/**
	 * Check template built-with.
	 *
	 * @param string|int $post_id Post ID.
	 * @return string
	 */
	public static function template_built_with( $post_id = null ) {
		$built_with = false;

		$post = get_post( $post_id );
		if ( $post && self::CPT === $post->post_type ) {
			$built_with_elementor_meta = get_post_meta( $post->ID, '_elementor_edit_mode', true );
			$built_with_wpbakery_meta  = get_post_meta( $post->ID, '_wpb_vc_js_status', true );
			$active_builders           = cardealer_get_active_page_builders();

			if ( $built_with_elementor_meta && 'builder' === $built_with_elementor_meta && in_array( 'elementor', $active_builders, true ) ) {
				$built_with = 'elementor';
			} elseif ( $built_with_wpbakery_meta && true === filter_var( $built_with_wpbakery_meta, FILTER_VALIDATE_BOOLEAN ) && in_array( 'wpbakery', $active_builders, true ) ) {
				$built_with = 'wpbakery';
			}
		}

		return $built_with;
	}

	/**
	 * Check whether mobile.
	 *
	 * @return bool
	 */
	public static function is_mobile() {
		global $post;

		$is_mobile = false;

		if (
			wp_is_mobile()
			|| (
				$post
				&& ( class_exists( 'CDHL_CPT_Template' ) && CDHL_CPT_Template::CPT === $post->post_type )
				&& ( 'vehicle_detail_mobile' === get_post_meta( $post->ID, 'cardealer_template_type', true ) )
			)
		) {
			$is_mobile = true;
		}
		return $is_mobile;
	}

	/**
	 * Process content for export/import.
	 *
	 * Process the content and all the inner elements, and prepare all the
	 * elements data for export/import.
	 *
	 * @since 1.5.0
	 * @access protected
	 *
	 * @param array  $content A set of elements.
	 * @param string $method  Accepts either `on_export` to export data or
	 *                        `on_import` to import data.
	 *
	 * @return mixed Processed content data.
	 */
	protected function process_export_import_content( $content, $method ) {
		return Plugin::$instance->db->iterate_data(
			$content, function( $element_data ) use ( $method ) {
				$element = Plugin::$instance->elements_manager->create_element_instance( $element_data );

				// If the widget/element isn't exist, like a plugin that creates a widget but deactivated
				if ( ! $element ) {
					return null;
				}

				return $this->process_element_export_import_content( $element, $method );
			}
		);
	}

	/**
		 * Process single element content for export/import.
		 *
		 * Process any given element and prepare the element data for export/import.
		 *
		 * @since 1.5.0
		 * @access protected
		 *
		 * @param Controls_Stack $element
		 * @param string         $method
		 *
		 * @return array Processed element data.
		 */
		protected function process_element_export_import_content( Controls_Stack $element, $method ) {
			$element_data = $element->get_data();

			if ( method_exists( $element, $method ) ) {
				// TODO: Use the internal element data without parameters.
				$element_data = $element->{$method}( $element_data );
			}

			foreach ( $element->get_controls() as $control ) {
				$control_class = Plugin::$instance->controls_manager->get_control( $control['type'] );

				// If the control isn't exist, like a plugin that creates the control but deactivated.
				if ( ! $control_class ) {
					return $element_data;
				}

				if ( method_exists( $control_class, $method ) ) {
					$element_data['settings'][ $control['name'] ] = $control_class->{$method}( $element->get_settings( $control['name'] ), $control );
				}

				// On Export, check if the control has an argument 'export' => false.
				if ( 'on_export' === $method && isset( $control['export'] ) && false === $control['export'] ) {
					unset( $element_data['settings'][ $control['name'] ] );
				}
			}

			return $element_data;
		}

		/**
		 * Replace elements IDs.
		 *
		 * For any given Elementor content/data, replace the IDs with new randomly
		 * generated IDs.
		 *
		 * @since 1.0.0
		 * @access protected
		 *
		 * @param array $content Any type of Elementor data.
		 *
		 * @return mixed Iterated data.
		 */
		protected function replace_elements_ids( $content ) {
			return Plugin::$instance->db->iterate_data( $content, function( $element ) {
				$element['id'] = Utils::generate_random_string();

				return $element;
			} );
		}

}
new CDHL_CPT_Template();
