<?php
/**
 * Banner class.
 *
 * @package cardealer-helper/elementor
 * @since   5.0.0
 */

namespace Cdhl_Elementor\Widgets;

use Elementor\Utils;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;

use Cdhl_Elementor\Widget_Controller\Widget_Controller;
use Cdhl_Elementor\Group_Controls;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor banner widget.
 *
 * Elementor widget that displays an banner.
 *
 * @since 5.0.0
 */
class CDFS_Dealers_Element extends Widget_Controller {

	/**
	 * Widget slug
	 *
	 * @var string
	 */
	protected $widget_slug = 'cdfs-dealers';

	/**
	 * Widget icon
	 *
	 * @var string
	 */
	protected $widget_icon = 'cdhl-widget-icon';

	/**
	 * Widget keywords
	 *
	 * @var array
	 */
	protected $keywords = array( 'custom', 'dealer', 'dealer-list', 'list' );

	/**
	 * Retrieve the widget title.
	 *
	 * @since 5.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'CDFS Dealers', 'cardealer-helper' );
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 5.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {
		$cars_taxonomies     = $this->get_taxonomies( false );
		$list_columns_option = $this->get_list_columns();

		$this->start_controls_section(
			'section_content_general',
			array(
				'label' => esc_html__( 'General', 'cardealer-helper' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'car_filters',
			[
				'label'       => esc_html__( 'Select Filters', 'cardealer-helper' ),
				'type'        => Controls_Manager::SELECT2,
				'multiple'    => true,
				'options'     => $cars_taxonomies,
			]
		);

		$this->add_control(
			'list_columns',
			[
				'label'       => esc_html__( 'Select Columns', 'cardealer-helper' ),
				'type'        => Controls_Manager::SELECT2,
				'multiple'    => true,
				'options'     => $list_columns_option,
				'default'     => array(
					'dealer',
					'rating',
					'stock',
					'phone',
					'location',
				),
			]
		);

		$this->add_control(
			'show_keyword',
			[
				'type'         => Controls_Manager::SWITCHER,
				'label'        => esc_html__( 'Show Keyword Field', 'cardealer-helper' ),
				'label_on'     => esc_html__( 'Yes', 'cardealer-helper' ),
				'label_off'    => esc_html__( 'No', 'cardealer-helper' ),
				'return_value' => 'yes',
				'default'      => 'no',
			]
		);

		$this->add_control(
			'show_location',
			[
				'type'         => Controls_Manager::SWITCHER,
				'label'        => esc_html__( 'Show Location Field', 'cardealer-helper' ),
				'description'  => esc_html__( 'Select this option to display the location search field to show the dealer\'s distance.', 'cardealer-helper' ),
				'label_on'     => esc_html__( 'Yes', 'cardealer-helper' ),
				'label_off'    => esc_html__( 'No', 'cardealer-helper' ),
				'return_value' => 'yes',
				'default'      => 'no',
			]
		);

		$this->add_control(
			'distance_unit',
			[
				'type'        => Controls_Manager::SELECT,
				'label'       => esc_html__( 'Distance Unit', 'cardealer-helper' ),
				'description' => esc_html__( 'Select a unit for the distance between dealer location and searched location.', 'cardealer-helper' ),
				'options'     => $this->get_distance_unit(),
				'default'     => 'km',
				'condition' => [
					'show_location' => 'yes',
				],
			]
		);


		$this->end_controls_section();
	}

	public function get_taxonomies( $flip_array = true ): array {
		$cars_taxonomy = cdhl_get_cars_taxonomy();
		if ( ! $flip_array ) {
			$cars_taxonomy = array_flip( $cars_taxonomy );
		}
		return $cars_taxonomy;
	}

	public function get_list_columns(): array {
		return array(
			'dealer'   => esc_html__( 'Dealer', 'cardealer-helper' ),
			'rating'   => esc_html__( 'Rating', 'cardealer-helper' ),
			'stock'    => esc_html__( 'Cars in Stock', 'cardealer-helper' ),
			'phone'    => esc_html__( 'Phone', 'cardealer-helper' ),
			'email'    => esc_html__( 'Email', 'cardealer-helper' ),
			'whatsapp' => esc_html__( 'WhatsApp', 'cardealer-helper' ),
			'location' => esc_html__( 'Location', 'cardealer-helper' ),
		);
	}

	public function get_orderby_options(): array {
		return array(
			'name'   => esc_html__( 'Dealer Name', 'cardealer-helper' ),
			'rating' => esc_html__( 'Rating', 'cardealer-helper' ),
			'date'   => esc_html__( 'Date', 'cardealer-helper' ),
			'stock'  => esc_html__( 'Cars in Stock', 'cardealer-helper' ),
		);
	}

	public function get_distance_unit( $type = 'full' ): array {
		$distance_units = array(
			'km'  => array(
				'key'   => 'km',
				'short' => esc_html__( 'KM', 'cardealer-helper' ),
				'full'  => esc_html__( 'Kilometer', 'cardealer-helper' ),
			),
			'mi'  => array(
				'key'   => 'mi',
				'short' => esc_html__( 'MI', 'cardealer-helper' ),
				'full'  => esc_html__( 'Miles', 'cardealer-helper' ),
			),
			'nmi' => array(
				'key'   => 'nmi',
				'short' => esc_html__( 'NMI', 'cardealer-helper' ),
				'full'  => esc_html__( 'Nautical mile', 'cardealer-helper' ),
			),
		);

		$vtype = ( 'short' === $type ) ? 'short' : 'full';

		return array_column( $distance_units, $vtype, 'key' );
	}

}
