<?php
/**
 * Elementor Vehicle Tabs class.
 *
 * @package cardealer-helper/elementor
 * @since   5.0.0
 */

namespace Cdhl_Elementor\Widgets;

use Elementor\Utils;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Stroke;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;

use Cdhl_Elementor\Widget_Controller\Widget_Controller;
use Cdhl_Elementor\Group_Controls;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor vehicle tabs widget.
 *
 * Elementor widget that displays vehicle tabs.
 *
 * @since 5.0.0
 */
class Vehicle_Tabs extends Widget_Controller {

	/**
	 * Widget slug
	 *
	 * @var string
	 */
	protected $widget_slug = 'vehicle-tabs';

	/**
	 * Widget icon
	 *
	 * @var string
	 */
	protected $widget_icon = 'cdhl-widget-icon';

	/**
	 * Widget keywords
	 *
	 * @var array
	 */
	protected $keywords = array( 'cars', 'vehicle', 'detail', 'single', 'tabs' );

	/**
	 * Retrieve the widget title.
	 *
	 * @since 5.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Vehicle Tabs', 'cardealer-helper' );
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 5.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( CDHL_ELEMENTOR_VEHIVLE_SINGLE_CAT );
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 5.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'section_tabs',
			[
				'label' => esc_html__( 'Tabs', 'cardealer-helper' ),
			]
		);

		$repeater = new \Elementor\Repeater();

		$default_tabs = apply_filters( 'cardealer_default_vehicle_tabs', array() );
		$vehicle_tabs = apply_filters( 'cardealer_vehicle_tabs_option', array() );
		$vehicle_tabs = array_merge(
			$vehicle_tabs,
			array(
				'custom' => 'Custom',
			)
		);

		$repeater->add_control(
			'selected_tab',
			[
				'label'   => esc_html__( 'Tab', 'cardealer-helper' ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => $vehicle_tabs,
			]
		);

		$repeater->add_control(
			'builtin_tab_notice',
			[
				'label'           => '',
				'type'            => \Elementor\Controls_Manager::RAW_HTML,
				'raw'             => esc_html__( 'The built-in tabs depend on the content so that they will appear according to the data.', 'cardealer-helper' ),
				'content_classes' => 'cardealer-elementor-panel-notice',
				'condition'   => array(
					'selected_tab!' => 'custom',
				),
			]
		);

		$repeater->add_control(
			'custom_tab_notice',
			[
				'label'           => esc_html__( 'Important Note', 'cardealer-helper' ),
				'type'            => \Elementor\Controls_Manager::RAW_HTML,
				'raw'             => esc_html__( 'If a custom tab is missing a title or content, the tab will not display on the front.', 'cardealer-helper' ),
				'content_classes' => 'cardealer-elementor-panel-notice',
				'condition'   => array(
					'selected_tab' => 'custom',
				),
			]
		);

		$repeater->add_control(
			'tab_title',
			[
				'label'       => esc_html__( 'Tab Title', 'cardealer-helper' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'Tab Title', 'cardealer-helper' ),
				'placeholder' => esc_html__( 'Tab Title', 'cardealer-helper' ),
				'label_block' => true,
			]
		);

		$repeater->add_control(
			'tab_icon',
			[
				'label'            => esc_html__( 'Icon', 'cardealer-helper' ),
				'type'             => Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'skin'             => 'inline',
				'label_block'      => false,
				'exclude_inline_options'=> array(
					'svg'
				),
			]
		);

		$repeater->add_control(
			'tab_content',
			[
				'label'       => esc_html__( 'Content', 'cardealer-helper' ),
				'type'        => Controls_Manager::WYSIWYG,
				'default'     => esc_html__( 'Tab Content', 'cardealer-helper' ),
				'placeholder' => esc_html__( 'Tab Content', 'cardealer-helper' ),
				'condition'   => array(
					'selected_tab' => 'custom',
				),
			]
		);

		$this->add_control(
			'tabs',
			[
				'label' => esc_html__( 'Tabs', 'cardealer-helper' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => array(
					array(
						'selected_tab' => 'overview',
						'tab_title'    => ( isset( $default_tabs['overview']['title'] ) && ! empty( $default_tabs['overview']['title'] ) ) ? $default_tabs['overview']['title'] : esc_html__( 'Overview', 'cardealer-helper' ),
						'tab_icon'     => array(
							'value'   => 'fas fa-sliders-h',
							'library' => 'fa-solid',
						),
					),
					array(
						'selected_tab' => 'features',
						'tab_title'    => ( isset( $default_tabs['features']['title'] ) && ! empty( $default_tabs['features']['title'] ) ) ? $default_tabs['features']['title'] : esc_html__( 'Features & Options', 'cardealer-helper' ),
						'tab_icon'     => array(
							'value'   => 'fas fa-list',
							'library' => 'fa-solid',
						),
					),
					array(
						'selected_tab' => 'technical',
						'tab_title'    => ( isset( $default_tabs['technical']['title'] ) && ! empty( $default_tabs['technical']['title'] ) ) ? $default_tabs['technical']['title'] : esc_html__( 'Technical', 'cardealer-helper' ),
						'tab_icon'     => array(
							'value'   => 'fas fa-cogs',
							'library' => 'fa-solid',
						),
					),
					array(
						'selected_tab' => 'general_info',
						'tab_title'    => ( isset( $default_tabs['general_info']['title'] ) && ! empty( $default_tabs['general_info']['title'] ) ) ? $default_tabs['general_info']['title'] : esc_html__( 'General Information', 'cardealer-helper' ),
						'tab_icon'     => array(
							'value'   => 'fas fa-info-circle',
							'library' => 'fa-solid',
						),
					),
					array(
						'selected_tab' => 'location',
						'tab_title'    => ( isset( $default_tabs['location']['title'] ) && ! empty( $default_tabs['location']['title'] ) ) ? $default_tabs['location']['title'] : esc_html__( 'Location', 'cardealer-helper' ),
						'tab_icon'     => array(
							'value'   => 'fas fa-map-marker-alt',
							'library' => 'fa-solid',
						),
					),
				),
				'title_field' => '{{{ get_selected_tab_label(selected_tab, obj) }}}',
			]
		);

		$this->end_controls_section();

	}
}
