<?php
/**
 * CDHL WooCommerce Integration class.
 *
 * @package  CDHL/Classes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

defined( 'ABSPATH' ) || exit;

/**
 * CDHL_Woocommerce_Integration class.
 */
class CDHL_Woocommerce_Integration {

	/**
	 * Constructor for the class.
	 */
	public function __construct() {
		// Set custom data store for "cars" CPT.
		add_filter( 'woocommerce_data_stores', array( $this, 'woocommerce_data_stores' ) );

		add_filter( 'woocommerce_add_cart_item_data', array( $this, 'add_cart_item_data' ), 10, 4 );                             // Step 2: Add Customer Data to WooCommerce Cart.
		add_filter( 'woocommerce_remove_cart_item', array( $this, 'remove_cart_item_data' ), 10, 4 );                            // Step 2: Add Customer Data to WooCommerce Cart.
		add_filter( 'woocommerce_get_item_data', array( $this, 'add_item_meta' ), 10, 2 );                                       // Step 3: Display meta details in the cart and checkout page.
		add_action( 'woocommerce_checkout_create_order_line_item', array( $this, 'add_checkout_order_line_item_meta' ), 10, 4 ); // Step 4: Add Custom Details as Order Line Items.
		add_filter( 'woocommerce_order_item_display_meta_key', array( $this, 'display_order_item_meta_key' ), 10, 3 );           // Step 5: Display meta details on order details page and in backend order.
		add_filter( 'woocommerce_order_item_display_meta_value', array( $this, 'display_order_item_meta_value' ), 10, 3 );
		add_filter( 'woocommerce_hidden_order_itemmeta', array( $this, 'hidden_order_itemmeta' ), 99 );

		// Add/Update stock validation.
		add_filter( 'woocommerce_add_to_cart_validation', array( $this, 'add_to_cart_validation' ), 10, 3 );
		add_filter( 'woocommerce_update_cart_validation', array( $this, 'update_cart_validation' ), 10, 4 );
		add_filter( 'woocommerce_cart_item_quantity', array( $this, 'limit_cart_item_quantity' ), 9999, 3 );

		add_action( 'woocommerce_checkout_order_created', array( $this, 'set_product_id' ) );

		// Old (discontinued) order data update method.
		// add_action( 'woocommerce_checkout_update_order_meta', array( $this, 'update_order_meta_before_checkout' ), 10, 2 );

		// Update vehicle stock once order status get completed.
		add_action( 'woocommerce_payment_complete', array( $this, 'order_status_changed_inward' ) );
		add_action( 'woocommerce_order_status_on-hold', array( $this, 'order_status_changed_inward' ) );
		add_action( 'woocommerce_order_status_processing', array( $this, 'order_status_changed_inward' ) );
		add_action( 'woocommerce_order_status_completed', array( $this, 'order_status_changed_inward' ) );

		// Update vehicle stock once order status get canceled refunded and failed.
		add_action( 'woocommerce_order_status_pending', array( $this, 'order_status_changed_outward' ) );
		add_action( 'woocommerce_order_status_failed', array( $this, 'order_status_changed_outward' ) );
		add_action( 'woocommerce_order_status_cancelled', array( $this, 'order_status_changed_outward' ) );
		add_action( 'woocommerce_order_status_refunded', array( $this, 'order_status_changed_outward' ) );

		add_action( 'cardealer_wc_cars_order_status_changed_inward_sell_vehicle', array( $this, 'decrease_cars_stock_on_order_inward' ), 10, 4 );
		add_action( 'cardealer_wc_cars_order_status_changed_outward_sell_vehicle', array( $this, 'increase_cars_stock_on_order_outward' ), 10, 4 );
	}

	/**
	 * Sell vehicle custom call
	 */
	public function woocommerce_data_stores ( $stores ) {
		require_once trailingslashit( CDHL_PATH ) . 'includes/woocommerce/class-cardealer-data-store-cpt.php';
		$stores['product'] = 'CarDealer_Data_Store_CPT';
		return $stores;
	}

	/**
	 * Add custom data to Cart
	 *
	 * @param array      $cart_item_data  Array of cart item data.
	 * @param string/int $product_id      Product ID.
	 * @param string/int $variation_id    Variation ID.
	 * @param string/int $quantity        Quantity.
	 * @return array
	 */
	public function add_cart_item_data( $cart_item_data, $product_id, $variation_id, $quantity ) {
		$post = get_post( $product_id );
		if ( $post && 'cars' === $post->post_type ) {
			$cardealer_order_types = cardealer_order_types();
			$cardealer_order_type  = ( isset( $_GET['cardealer_order_type'] ) && ! empty( $_GET['cardealer_order_type'] ) && array_key_exists( $_GET['cardealer_order_type'], $cardealer_order_types ) ) ? $_GET['cardealer_order_type'] : 'sell_vehicle';

			$cart_item_data['cardealer_order_type']  = $cardealer_order_type;
			$cart_item_data['_cd_vehicle_id']        = $product_id;

			$cart_item_data = apply_filters( 'cardealer_order_add_cart_item_data', $cart_item_data, $product_id, $variation_id, $quantity, $cardealer_order_type );
			$cart_item_data = apply_filters( 'cardealer_order_add_cart_item_data_' . $cardealer_order_type, $cart_item_data, $product_id, $variation_id, $quantity );

			do_action( 'cardealer_order_after_added_cart_item_data', $cart_item_data, $product_id, $variation_id, $quantity, $cardealer_order_type );
			do_action( 'cardealer_order_after_added_cart_item_data_' . $cardealer_order_type, $cart_item_data, $product_id, $variation_id, $quantity );
		}
		return $cart_item_data;
	}

	/**
	 * Process data before item is being removed from Cart.
	 *
	 * @param string $cart_item_key Cart item key to remove from the cart.
	 * @param WC_Cart $cart         Cart object to remove the item.
	 */
	public function remove_cart_item_data( $cart_item_key, $wc_cart ) {
		$cart_item_data        = $wc_cart->get_cart_item( $cart_item_key );
		$vehicle_id            = isset( $cart_item_data['_cd_vehicle_id'] ) ? $cart_item_data['_cd_vehicle_id'] : '';
		$cardealer_order_types = cardealer_order_types();
		$cardealer_order_type  = ( isset( $cart_item_data['cardealer_order_type'] ) && ! empty( $cart_item_data['cardealer_order_type'] ) && array_key_exists( $cart_item_data['cardealer_order_type'], $cardealer_order_types ) ) ? $cart_item_data['cardealer_order_type'] : 'sell_vehicle';

		if ( ! empty( $vehicle_id ) ) {
			do_action( 'cardealer_before_remove_cart_item', $cart_item_key, $cart_item_data, $wc_cart, $vehicle_id, $cardealer_order_type );
			do_action( 'cardealer_before_remove_cart_item_' . $cardealer_order_type, $cart_item_key, $cart_item_data, $wc_cart, $vehicle_id  );
		}
	}

	/**
	 * Display meta details in the cart and checkout page.
	 *
	 * @param array $item_data   Item data.
	 * @param array $cart_item   Cart item.
	 * @return array
	 */
	public function add_item_meta( $item_data, $cart_item ) {
		$post = get_post( $cart_item['product_id'] );
		if ( $post && 'cars' === $post->post_type && array_key_exists( 'cardealer_order_type', $cart_item ) ) {
			$cardealer_order_types = cardealer_order_types();

			$item_data[] = array(
				'key'     => esc_html__( 'Order Type', 'cardealer-helper' ),
				'value'   => wc_clean( $cart_item['cardealer_order_type'] ),
				'display' => $cardealer_order_types[ $cart_item['cardealer_order_type'] ],
			);
		}

		return $item_data;
	}

	/**
	 * Add checkout order line item meta.
	 *
	 * @param WC_Order_Item_Product $item           WC_Order_Item_Product instance.
	 * @param string                $cart_item_key  Cart item key.
	 * @param string                $values         Cart item value.
	 * @param WC_Order              $order          Order instance.
	 * @return void
	 */
	public function add_checkout_order_line_item_meta( $item, $cart_item_key, $values, $order ) {
		$post = get_post( $values['product_id'] );
		if ( $post && 'cars' === $post->post_type ) {
			$item->add_meta_data( '_cd_vehicle_id', $values['product_id'] );
			if ( array_key_exists( 'cardealer_order_type', $values ) ) {
				$item->add_meta_data( 'cardealer_order_type', $values['cardealer_order_type'] );
			}
			do_action( 'cardealer_add_checkout_order_line_item_meta', $item, $cart_item_key, $values, $order, $values['cardealer_order_type'] );
			do_action( 'cardealer_add_checkout_order_line_item_meta_' . $values['cardealer_order_type'], $item, $cart_item_key, $values, $order );
		}
	}

	/**
	 * Display order item meta key.
	 *
	 * @param string        $display_key  Display key.
	 * @param string        $meta         Order meta.
	 * @param WC_Order_Item $rder_item    Order instance.
	 * @return string
	 */
	public function display_order_item_meta_key( $display_key, $meta, $rder_item ) {
		if ( 'cardealer_order_type' === $display_key ) {
			$display_key = esc_html__( 'Order Type', 'cardealer-helper' );
		}

		return $display_key;
	}

	/**
	 * Display order item meta value.
	 *
	 * @param string        $display_key  Display key.
	 * @param string        $meta         Order meta.
	 * @param WC_Order_Item $rder_item    Order instance.
	 * @return string
	 */
	public function display_order_item_meta_value( $display_value, $meta, $rder_item ) {
		$cardealer_order_types = cardealer_order_types();
		$display_value   = array_key_exists( $display_value, $cardealer_order_types ) ? $cardealer_order_types[ $display_value ] : $display_value;
		return $display_value;
	}

	public function hidden_order_itemmeta( $itemmetas ) {
		// $itemmetas[] = '_cd_vehicle_id';
		return $itemmetas;
	}

	/**
	 * Prevent vehicle out stock being added to the cart.
	 *
	 * @param  bool $passed     Validation.
	 * @param  int  $product_id Product ID.
	 * @return bool
	 */
	public function add_to_cart_validation( $passed, $product_id, $quantity_to_add ) {
		$post = get_post( $product_id );

		if ( $post && 'cars' === $post->post_type ) {
			$cardealer_order_types = cardealer_order_types();
			$cardealer_order_type  = ( isset( $_GET['cardealer_order_type'] ) && ! empty( $_GET['cardealer_order_type'] ) && array_key_exists( $_GET['cardealer_order_type'], $cardealer_order_types ) ) ? $_GET['cardealer_order_type'] : 'sell_vehicle';
			$stock_quantity  = 1;
			$cart_quantity   = 0;

			if ( 'sell_vehicle' === $cardealer_order_type ) {
				$stock_quantity = cardealer_get_vehicle_stock( $product_id );
			}

			$cart_items = WC()->cart->get_cart();

			foreach ( $cart_items as $cart_item_key => $cart_item ) {
				if (
					( (int) $product_id === (int) $cart_item['product_id'] )
					&& ( isset( $cart_item['cardealer_order_type'] ) && $cardealer_order_type === $cart_item['cardealer_order_type'] )
				) {
					$cart_quantity = $cart_item['quantity'];
					break;
				}
			}
			$new_quantity = $cart_quantity + $quantity_to_add;
			$passed = $this->cart_validation_callback( $passed, $stock_quantity, $cart_quantity, $new_quantity, $post );
			// die();
		}

		return $passed;
	}

	/**
	 * Prevent vehicle out stock being Update cart.
	 *
	 * @param  bool $passed     Validation.
	 * @param  int  $product_id Product ID.
	 * @return bool
	 */
	public function update_cart_validation( $passed, $cart_item_key, $values, $cart_quantity ) {
		$post = get_post( $values['product_id'] );

		if ( $post && 'cars' === $post->post_type ) {
			$cardealer_order_types = cardealer_order_types();
			$cardealer_order_type  = ( isset( $values['cardealer_order_type'] ) && ! empty( $values['cardealer_order_type'] ) && array_key_exists( $values['cardealer_order_type'], $cardealer_order_types ) ) ? $values['cardealer_order_type'] : 'sell_vehicle';
			$stock_quantity  = 1;
			$new_quantity    = $cart_quantity;
			$cart_quantity   = $values['quantity'];

			if ( ! isset( $values['cardealer_order_type'] ) || 'sell_vehicle' === $values['cardealer_order_type'] ) {
				$stock_quantity = cardealer_get_vehicle_stock( $values['product_id'] );
			}

			$passed = $this->cart_validation_callback( $passed, $stock_quantity, $cart_quantity, $new_quantity, $post );
		}

		return $passed;
	}

	public function cart_validation_callback( $passed, $stock_quantity, $cart_quantity, $new_quantity, $post ) {
		// var_dump( $passed, $stock_quantity, $cart_quantity, $new_quantity );
		if ( $stock_quantity < 1 ) {
			$passed = false;
			wc_add_notice( esc_html__( 'Sorry, the item is out of stock and cannot be added to the cart.', 'cardealer-helper' ), 'error' );
		} elseif ( (int) $new_quantity > 1 && ! apply_filters( 'cardealer_allow_same_car_more_than_one_per_order', false ) ) {
			$passed = false;
			wc_add_notice( esc_html__( 'Sorry, only one copy of this product is allowed to be added to the cart.', 'cardealer-helper' ), 'error' );
		}elseif ( $new_quantity > $stock_quantity ) {
			/* translators: 1: quantity in stock 2: current quantity */
			$message = sprintf(
				__( 'You cannot add that amount to the cart &mdash; we have %1$s in stock and you already have %2$s in your cart.', 'cardealer-helper' ),
				wc_format_stock_quantity_for_display( $stock_quantity, $post ),
				wc_format_stock_quantity_for_display( $cart_quantity, $post )
			);

			/**
			 * Filters message about product not having enough stock accounting for what's already in the cart.
			 *
			 * @param string  $message        Message.
			 * @param WP_Post $post           Product data.
			 * @param int     $stock_quantity Quantity remaining.
			 * @param int     $quantity       Requested quantity.
			 *
			 * @since 5.3.0
			 */
			$message = apply_filters( 'woocommerce_cart_product_not_enough_stock_already_in_cart_message', $message, $post, $stock_quantity, $cart_quantity );
			$passed  = false;
			wc_add_notice( $message, 'error' );
		}

		return $passed;
	}

	public function limit_cart_item_quantity( $product_quantity, $cart_item_key, $cart_item ) {
		$vehicle = get_post( $cart_item['product_id'] );
		global $product;

		if ( $vehicle && 'cars' === $vehicle->post_type ) {
			$cardealer_order_types  = cardealer_order_types();
			$max_value        = 1;
			$product_quantity = sprintf( '1 <input type="hidden" name="cart[%s][qty]" value="1" />', $cart_item_key );

			if ( isset( $cart_item['cardealer_order_type'] ) && ! empty( $cart_item['cardealer_order_type'] ) && array_key_exists( $cart_item['cardealer_order_type'], $cardealer_order_types ) ) {
				if ( 'sell_vehicle' === $cart_item['cardealer_order_type'] ) {
					$vehicle_stock = cardealer_get_vehicle_stock( $cart_item['product_id'] );
					$max_value     = $vehicle_stock;
					if ( apply_filters( 'cardealer_allow_same_car_more_than_one_per_order', false ) ) {
						$product_quantity = woocommerce_quantity_input(
							array(
								'input_name'   => "cart[{$cart_item_key}][qty]",
								'input_value'  => $cart_item['quantity'],
								'max_value'    => $max_value,
								'min_value'    => '0',
								'product_name' => get_the_title( $vehicle ),
							),
							null,
							false
						);
					}
				}
				/*
				elseif ( 'advertise_item' === $cart_item['cardealer_order_type'] ) {
					$max_value = 1;
				}
				*/
			}
		}

		return $product_quantity;
	}

	public function set_product_id( $order ) {
		foreach ( $order->get_items() as $item_id => $item ) {
			$cd_vehicle_id = $item->get_meta( '_cd_vehicle_id', true );
			$post          = get_post( $cd_vehicle_id );
			if ( $cd_vehicle_id && is_numeric( $cd_vehicle_id ) && $post && 'cars' === $post->post_type ) {
				wc_update_order_item_meta( $item_id, '_product_id', $cd_vehicle_id );
			}
		}
	}

	/**
	 * Add custom meta to order for vehicle sell
	 */
	/*
	public function update_order_meta_before_checkout( $order_id, $data ) {
		global $car_dealer_options;

		$cart_items = WC()->cart->get_cart();

		foreach ( $cart_items as $cart_item_key => $cart_item ) {
			$id          = $cart_item['product_id'];
			$post_object = get_post( $cart_item['product_id'] );

			if ( 'cars' !== $post_object->post_type ) {
				continue;
			}

			$sell_vehicle_status = get_post_meta( $id, 'sell_vehicle_status', true );
			$sell_vehicle_option = ( isset( $car_dealer_options['sell_vehicle_enable_all_option'] ) ) ? $car_dealer_options['sell_vehicle_enable_all_option'] : 0;

			if ( 'enable' === $sell_vehicle_status || $sell_vehicle_option ) {
				$vehicle_old_ids = get_post_meta( $order_id, 'vehicle_ids', true );
				if ( ! empty( $vehicle_old_ids ) ) {
					$vehicle_ids   = $vehicle_old_ids;
					$vehicle_ids[] = $id;
				} else {
					$vehicle_ids[] = $id;
				}

				update_post_meta( $order_id, 'vehicle_ids', $vehicle_ids );
			}
		}

		return true;
	}
	*/

	/**
	 * Update vehicle stock once order status get completed
	 */
	public function order_status_changed_inward( $order_id ) {

		if ( metadata_exists( 'post', $order_id, 'vehicle_ids' ) ) {
			$this->order_status_changed_inward_legacy( $order_id );
		} else {
			$this->order_status_changed_inward_new( $order_id );
		}
	}

	public function order_status_changed_inward_legacy( $order_id ) {
		$vehicle_ids           = get_post_meta( $order_id, 'vehicle_ids', true );
		$vehicle_stock_reduced = get_post_meta( $order_id, 'vehicle_stock_reduced', true );

		if ( $vehicle_ids && ! $vehicle_stock_reduced ) {
			foreach ( $vehicle_ids as $vehicle_id ) {
				$vehicle_in_stock = cardealer_get_vehicle_stock( $vehicle_id );
				if ( $vehicle_in_stock > 0 ) {

					if ( 1 === $vehicle_in_stock ) {
						update_post_meta( $vehicle_id, 'car_status', 'sold' );
					}

					update_post_meta( $vehicle_id, 'total_vehicle_in_stock', $vehicle_in_stock - 1 );
				}
			}
			update_post_meta( $order_id, 'vehicle_stock_reduced', true );
		}
	}

	public function order_status_changed_inward_new( $order_id ) {
		$action = current_action();
		$order  = wc_get_order( $order_id );

		foreach ( $order->get_items() as $item_id => $item ) {
			$cd_vehicle_id = $item->get_meta( '_cd_vehicle_id', true );
			$post          = get_post( $cd_vehicle_id );

			if ( $cd_vehicle_id && is_numeric( $cd_vehicle_id ) && $post && 'cars' === $post->post_type ) {
				$cd_item_type    = $item->get_meta( 'cardealer_order_type', true );
				$cardealer_order_types = cardealer_order_types();
				$cardealer_order_type  = ( ! empty( $cd_item_type ) && array_key_exists( $cd_item_type, $cardealer_order_types ) ) ? $cd_item_type : 'sell_vehicle';

				do_action( "cardealer_wc_cars_order_status_changed_inward_{$cardealer_order_type}", $item_id, $item, $order, $cd_vehicle_id, $action );
			}
		}
	}

	/**
	 * Update vehicle stock once order status get canceled refunded and failed
	 */
	public function order_status_changed_outward( $order_id ) {
		if ( metadata_exists( 'post', $order_id, 'vehicle_ids' ) ) {
			$this->order_status_changed_outward_legacy( $order_id );
		} else {
			$this->order_status_changed_outward_new( $order_id );
		}
	}
	public function order_status_changed_outward_legacy( $order_id ) {
		$vehicle_ids           = get_post_meta( $order_id, 'vehicle_ids', true );
		$vehicle_stock_reduced = get_post_meta( $order_id, 'vehicle_stock_reduced', true );

		if ( $vehicle_ids && $vehicle_stock_reduced ) {
			foreach ( $vehicle_ids as $vehicle_id ) {
				$vehicle_in_stock = cardealer_get_vehicle_stock( $vehicle_id );

				if ( is_numeric( $vehicle_in_stock ) ) {
					update_post_meta( $vehicle_id, 'car_status', 'unsold' );
					update_post_meta( $vehicle_id, 'total_vehicle_in_stock', $vehicle_in_stock + 1 );
				}
			}
			delete_post_meta( $order_id, 'vehicle_stock_reduced' );
		}
	}

	public function order_status_changed_outward_new( $order_id ) {
		$action = current_action();
		$order  = wc_get_order( $order_id );

		foreach ( $order->get_items() as $item_id => $item ) {
			$cd_vehicle_id = $item->get_meta( '_cd_vehicle_id', true );
			$post          = get_post( $cd_vehicle_id );

			if ( $cd_vehicle_id && is_numeric( $cd_vehicle_id ) && $post && 'cars' === $post->post_type ) {
				$cd_item_type    = $item->get_meta( 'cardealer_order_type', true );
				$cardealer_order_types = cardealer_order_types();
				$cardealer_order_type  = ( ! empty( $cd_item_type ) && array_key_exists( $cd_item_type, $cardealer_order_types ) ) ? $cd_item_type : 'sell_vehicle';

				do_action( "cardealer_wc_cars_order_status_changed_outward_{$cardealer_order_type}", $item_id, $item, $order, $cd_vehicle_id, $action );
			}
		}
	}

	public function decrease_cars_stock_on_order_inward( $item_id, $item, $order, int $vehicle_id ) {
		$qty                   = (int) $item->get_quantity();
		$vehicle_stock_reduced = $item->get_meta( '_vehicle_stock_reduced', true );

		if ( $vehicle_id && $qty > 0 && ! $vehicle_stock_reduced ) {
			$vehicle_in_stock = (int) cardealer_get_vehicle_stock( $vehicle_id );
			$new_stock        = $vehicle_in_stock - $qty;

			$stock_update_status = update_post_meta( $vehicle_id, 'total_vehicle_in_stock', $new_stock );

			if ( $stock_update_status ) {
				if ( $new_stock <= 0 ) {
					update_post_meta( $vehicle_id, 'car_status', 'sold' );
				}

				$item->add_meta_data( '_vehicle_stock_reduced', $qty, true );
				$item->save();
			}
		}
	}

	public function increase_cars_stock_on_order_outward( $item_id, $item, $order, int $vehicle_id ) {
		$qty                   = (int) $item->get_quantity();
		$vehicle_stock_reduced = $item->get_meta( '_vehicle_stock_reduced', true );

		if ( $vehicle_id && $qty > 0 && $vehicle_stock_reduced ) {
			$vehicle_in_stock = (int) cardealer_get_vehicle_stock( $vehicle_id );
			$new_stock        = $vehicle_in_stock + $qty;

			$stock_update_status = update_post_meta( $vehicle_id, 'total_vehicle_in_stock', $new_stock );

			if ( $stock_update_status ) {
				if ( $new_stock > 0 ) {
					update_post_meta( $vehicle_id, 'car_status', 'unsold' );
				}

				$item->update_meta_data( '_vehicle_stock_reduced', 0, true );
				$item->save();
			}
		}
	}

}
new CDHL_Woocommerce_Integration();
